@extends('layouts.app')

@section('title', 'My Reviews')

@section('content')
<div class="container mt-4">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2>
                    <i class="fas fa-star text-warning me-2"></i>
                    My Reviews
                </h2>
                
                <!-- Filter Dropdown -->
                <div class="dropdown">
                    <button class="btn btn-outline-primary dropdown-toggle" type="button" 
                            data-bs-toggle="dropdown">
                        <i class="fas fa-filter me-1"></i>
                        Filter by Rating
                    </button>
                    <ul class="dropdown-menu">
                        <li><a class="dropdown-item" href="{{ route('review.my-reviews') }}">All Reviews</a></li>
                        <li><hr class="dropdown-divider"></li>
                        @for($i = 5; $i >= 1; $i--)
                            <li>
                                <a class="dropdown-item {{ request('rating') == $i ? 'active' : '' }}" 
                                   href="{{ request()->fullUrlWithQuery(['rating' => $i]) }}">
                                    {{ $i }} Star{{ $i > 1 ? 's' : '' }}
                                </a>
                            </li>
                        @endfor
                    </ul>
                </div>
            </div>

            @if($reviews->count() > 0)
                <!-- Reviews Summary -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <h4 class="text-primary">{{ $reviews->total() }}</h4>
                                <small class="text-muted">Total Reviews</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <h4 class="text-success">{{ number_format($reviews->avg('rating'), 1) }}</h4>
                                <small class="text-muted">Average Rating</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <h4 class="text-warning">{{ $reviews->where('rating', '>=', 4)->count() }}</h4>
                                <small class="text-muted">Positive Reviews</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <h4 class="text-info">{{ $reviews->where('created_at', '>=', now()->subDays(30))->count() }}</h4>
                                <small class="text-muted">This Month</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Reviews List -->
                <div class="reviews-list">
                    @foreach($reviews as $review)
                        <div class="card mb-3">
                            <div class="card-body">
                                <!-- Review Header -->
                                <div class="d-flex justify-content-between align-items-start mb-3">
                                    <div class="d-flex align-items-center">
                                        <div class="me-3">
                                            <div class="bg-primary rounded-circle d-inline-flex align-items-center justify-content-center" 
                                                 style="width: 50px; height: 50px;">
                                                <span class="text-white">{{ substr($review->reviewee->name, 0, 1) }}</span>
                                            </div>
                                        </div>
                                        <div>
                                            <h6 class="mb-1">{{ $review->reviewee->name }}</h6>
                                            <small class="text-muted">
                                                {{ $review->reviewee->user_type === 'professional' ? 'Professional' : 'Client' }}
                                            </small>
                                            <div class="star-display">
                                                @for($i = 1; $i <= 5; $i++)
                                                    <i class="fas fa-star {{ $i <= $review->rating ? 'text-warning' : 'text-light' }}"></i>
                                                @endfor
                                                <span class="text-muted ms-2">{{ $review->created_at->format('M d, Y') }}</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Actions -->
                                    <div class="dropdown">
                                        <button class="btn btn-sm btn-outline-secondary dropdown-toggle" 
                                                type="button" data-bs-toggle="dropdown">
                                            <i class="fas fa-ellipsis-h"></i>
                                        </button>
                                        <ul class="dropdown-menu">
                                            <li>
                                                <a class="dropdown-item" href="{{ route('review.show', $review) }}">
                                                    <i class="fas fa-eye me-2"></i>View Details
                                                </a>
                                            </li>
                                            @if($review->created_at->diffInHours(now()) <= 24)
                                                <li>
                                                    <button class="dropdown-item" onclick="editReview({{ $review->id }})">
                                                        <i class="fas fa-edit me-2"></i>Edit Review
                                                    </button>
                                                </li>
                                            @endif
                                            <li>
                                                <a class="dropdown-item" href="{{ route('consultation.show', $review->consultation) }}">
                                                    <i class="fas fa-calendar me-2"></i>View Consultation
                                                </a>
                                            </li>
                                        </ul>
                                    </div>
                                </div>

                                <!-- Review Content -->
                                <div class="review-content mb-3">
                                    <p class="mb-0">{{ $review->comment }}</p>
                                </div>

                                <!-- Review Metadata -->
                                <div class="review-metadata">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-calendar me-1"></i>
                                                Consultation: {{ $review->consultation->scheduled_at->format('M d, Y') }}
                                            </small>
                                        </div>
                                        <div class="col-md-6">
                                            <small class="text-muted">
                                                <i class="fas fa-clock me-1"></i>
                                                Duration: {{ $review->consultation->duration }} minutes
                                            </small>
                                        </div>
                                    </div>
                                    
                                    @if($review->is_anonymous)
                                        <div class="mt-2">
                                            <span class="badge bg-secondary">
                                                <i class="fas fa-user-secret me-1"></i>Anonymous Review
                                            </span>
                                        </div>
                                    @endif
                                    
                                    @if($review->created_at != $review->updated_at)
                                        <div class="mt-2">
                                            <small class="text-info">
                                                <i class="fas fa-edit me-1"></i>
                                                Updated on {{ $review->updated_at->format('M d, Y') }}
                                            </small>
                                        </div>
                                    @endif
                                </div>

                                <!-- Review Status -->
                                <div class="mt-3 pt-2 border-top">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <span class="badge bg-{{ $review->status === 'active' ? 'success' : 'warning' }}">
                                            {{ ucfirst($review->status) }}
                                        </span>
                                        
                                        @if($review->created_at->diffInHours(now()) <= 24)
                                            <small class="text-success">
                                                <i class="fas fa-clock me-1"></i>
                                                Editable for {{ 24 - $review->created_at->diffInHours(now()) }} more hours
                                            </small>
                                        @else
                                            <small class="text-muted">
                                                <i class="fas fa-lock me-1"></i>
                                                No longer editable
                                            </small>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-center">
                    {{ $reviews->links() }}
                </div>

                <!-- Tips Section -->
                <div class="card mt-4 bg-light">
                    <div class="card-body">
                        <h6><i class="fas fa-lightbulb text-warning me-2"></i>Review Tips</h6>
                        <ul class="mb-0 small">
                            <li>You can edit your reviews within 24 hours of posting</li>
                            <li>Be honest and constructive in your feedback</li>
                            <li>Reviews help improve the quality of consultations on our platform</li>
                            <li>Both clients and professionals can leave reviews after each consultation</li>
                        </ul>
                    </div>
                </div>
            @else
                <!-- No Reviews -->
                <div class="text-center py-5">
                    <i class="fas fa-star fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">No reviews yet</h5>
                    <p class="text-muted">
                        You haven't left any reviews yet. After completing a consultation, you'll be able to leave a review.
                    </p>
                    
                    @if(Auth::user()->user_type === 'client')
                        <a href="{{ route('client.professionals') }}" class="btn btn-primary">
                            <i class="fas fa-search me-1"></i>Find Professionals
                        </a>
                    @else
                        <a href="{{ route('professional.dashboard') }}" class="btn btn-primary">
                            <i class="fas fa-tachometer-alt me-1"></i>Go to Dashboard
                        </a>
                    @endif
                </div>
            @endif
        </div>
    </div>
</div>

@push('styles')
<style>
.star-display {
    font-size: 0.875rem;
}

.review-content {
    line-height: 1.6;
}

.review-metadata {
    background-color: #f8f9fa;
    border-radius: 0.375rem;
    padding: 0.75rem;
}

.card {
    transition: box-shadow 0.2s;
}

.card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}
</style>
@endpush

@push('scripts')
<script>
function editReview(reviewId) {
    // In a real application, this would open an edit modal or redirect to edit page
    alert('Edit functionality would open here for review ID: ' + reviewId);
}
</script>
@endpush
@endsection