<?php

namespace App\Services;

use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\Professional;
use App\Models\Payment;
use Carbon\Carbon;

class SubscriptionService
{
    /**
     * Create a new subscription for a professional
     */
    public function createSubscription(Professional $professional, SubscriptionPlan $plan, $billingCycle = 'monthly', $paymentDetails = [])
    {
        $amount = $billingCycle === 'monthly' ? $plan->monthly_price : $plan->yearly_price;
        $duration = $billingCycle === 'monthly' ? 1 : 12; // months
        
        $subscription = Subscription::create([
            'professional_id' => $professional->id,
            'subscription_plan_id' => $plan->id,
            'billing_cycle' => $billingCycle,
            'amount_paid' => $amount,
            'start_date' => now(),
            'end_date' => now()->addMonths($duration),
            'status' => 'active',
            'payment_details' => $paymentDetails,
            'auto_renew' => true,
        ]);

        return $subscription;
    }

    /**
     * Upgrade a professional's subscription
     */
    public function upgradeSubscription(Professional $professional, SubscriptionPlan $newPlan, $billingCycle = 'monthly')
    {
        $currentSubscription = $professional->activeSubscription();
        
        if (!$currentSubscription) {
            return $this->createSubscription($professional, $newPlan, $billingCycle);
        }

        // Calculate prorated amount
        $proratedAmount = $this->calculateProratedAmount($currentSubscription, $newPlan, $billingCycle);
        
        // Cancel current subscription
        $currentSubscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => 'Upgraded to ' . $newPlan->name,
        ]);

        // Create new subscription
        $newSubscription = $this->createSubscription($professional, $newPlan, $billingCycle);
        
        return $newSubscription;
    }

    /**
     * Calculate prorated amount for subscription upgrade
     */
    private function calculateProratedAmount(Subscription $currentSubscription, SubscriptionPlan $newPlan, $billingCycle)
    {
        $remainingDays = now()->diffInDays($currentSubscription->end_date);
        $totalDays = $currentSubscription->billing_cycle === 'monthly' ? 30 : 365;
        
        $newAmount = $billingCycle === 'monthly' ? $newPlan->monthly_price : $newPlan->yearly_price;
        $currentAmount = $currentSubscription->amount_paid;
        
        $usedAmount = $currentAmount * (($totalDays - $remainingDays) / $totalDays);
        $refundAmount = $currentAmount - $usedAmount;
        
        $proratedAmount = $newAmount - $refundAmount;
        
        return max(0, $proratedAmount); // Ensure non-negative
    }

    /**
     * Check and update expired subscriptions
     */
    public function checkExpiredSubscriptions()
    {
        $expiredSubscriptions = Subscription::where('status', 'active')
            ->where('end_date', '<', now())
            ->get();

        foreach ($expiredSubscriptions as $subscription) {
            if ($subscription->auto_renew) {
                $this->renewSubscription($subscription);
            } else {
                $subscription->update(['status' => 'expired']);
            }
        }

        return $expiredSubscriptions->count();
    }

    /**
     * Renew a subscription
     */
    public function renewSubscription(Subscription $subscription)
    {
        $plan = $subscription->subscriptionPlan;
        $amount = $subscription->billing_cycle === 'monthly' ? $plan->monthly_price : $plan->yearly_price;
        $duration = $subscription->billing_cycle === 'monthly' ? 1 : 12;

        // Create payment record for renewal
        $payment = Payment::create([
            'payment_id' => Payment::generatePaymentId(),
            'user_id' => $subscription->professional->user_id,
            'payable_id' => $subscription->id,
            'payable_type' => Subscription::class,
            'payment_type' => 'subscription',
            'amount' => $amount,
            'gst_amount' => ($amount * 18) / 100, // 18% GST
            'total_amount' => $amount + (($amount * 18) / 100),
            'currency' => 'INR',
            'status' => 'pending',
            'payment_gateway' => 'razorpay', // Default gateway
        ]);

        // Update subscription
        $subscription->update([
            'start_date' => $subscription->end_date,
            'end_date' => $subscription->end_date->addMonths($duration),
            'amount_paid' => $amount,
            'status' => 'active',
        ]);

        return $subscription;
    }

    /**
     * Cancel a subscription
     */
    public function cancelSubscription(Subscription $subscription, $reason = null)
    {
        $subscription->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
            'cancellation_reason' => $reason,
            'auto_renew' => false,
        ]);

        return $subscription;
    }

    /**
     * Get subscription analytics for admin dashboard
     */
    public function getSubscriptionAnalytics()
    {
        $totalSubscriptions = Subscription::count();
        $activeSubscriptions = Subscription::active()->count();
        $expiredSubscriptions = Subscription::where('status', 'expired')->count();
        $cancelledSubscriptions = Subscription::where('status', 'cancelled')->count();

        $revenueThisMonth = Subscription::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount_paid');

        $subscriptionsByPlan = SubscriptionPlan::withCount(['activeSubscriptions'])
            ->get()
            ->map(function ($plan) {
                return [
                    'plan_name' => $plan->name,
                    'active_count' => $plan->active_subscriptions_count,
                    'revenue' => $plan->activeSubscriptions()->sum('amount_paid'),
                ];
            });

        return [
            'total_subscriptions' => $totalSubscriptions,
            'active_subscriptions' => $activeSubscriptions,
            'expired_subscriptions' => $expiredSubscriptions,
            'cancelled_subscriptions' => $cancelledSubscriptions,
            'revenue_this_month' => $revenueThisMonth,
            'subscriptions_by_plan' => $subscriptionsByPlan,
        ];
    }

    /**
     * Get commission rate for a professional based on their subscription
     */
    public function getCommissionRate(Professional $professional)
    {
        $subscription = $professional->activeSubscription();
        
        if (!$subscription) {
            return 50; // Default commission if no subscription
        }

        return $subscription->subscriptionPlan->commission_percentage;
    }
}