<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\Consultation;
use App\Models\Subscription;
use Exception;

class PaymentService
{
    /**
     * Create payment intent for Razorpay
     */
    public function createPaymentIntent($payable, $userId, $paymentType)
    {
        if ($payable instanceof Consultation) {
            $amount = $payable->total_amount;
            $description = "Consultation with " . $payable->professional->user->name;
        } elseif ($payable instanceof Subscription) {
            $amount = $payable->amount_paid + (($payable->amount_paid * 18) / 100); // Including GST
            $description = "Subscription: " . $payable->subscriptionPlan->name;
        } else {
            throw new Exception("Invalid payable type");
        }

        $payment = Payment::create([
            'payment_id' => Payment::generatePaymentId(),
            'user_id' => $userId,
            'payable_id' => $payable->id,
            'payable_type' => get_class($payable),
            'payment_type' => $paymentType,
            'amount' => $payable instanceof Consultation ? $payable->consultation_fee + $payable->platform_fee : $payable->amount_paid,
            'gst_amount' => $payable instanceof Consultation ? $payable->gst_amount : (($payable->amount_paid * 18) / 100),
            'total_amount' => $amount,
            'currency' => 'INR',
            'status' => 'pending',
            'payment_gateway' => 'razorpay',
        ]);

        // In a real application, you would integrate with Razorpay API here
        $razorpayOrder = $this->createRazorpayOrder($payment, $description);
        
        $payment->update([
            'gateway_order_id' => $razorpayOrder['id'],
        ]);

        return $payment;
    }

    /**
     * Mock Razorpay order creation
     */
    private function createRazorpayOrder($payment, $description)
    {
        // This is a mock implementation
        // In production, you would use Razorpay SDK:
        /*
        $api = new \Razorpay\Api\Api($keyId, $keySecret);
        $orderData = [
            'receipt' => $payment->payment_id,
            'amount' => $payment->total_amount * 100, // Amount in paise
            'currency' => 'INR',
            'notes' => [
                'description' => $description,
                'payment_id' => $payment->payment_id,
            ]
        ];
        return $api->order->create($orderData);
        */

        return [
            'id' => 'order_' . uniqid(),
            'amount' => $payment->total_amount * 100,
            'currency' => 'INR',
            'receipt' => $payment->payment_id,
        ];
    }

    /**
     * Process payment success
     */
    public function processPaymentSuccess(Payment $payment, $gatewayPaymentId, $gatewayResponse = [])
    {
        $payment->update([
            'status' => 'completed',
            'gateway_payment_id' => $gatewayPaymentId,
            'gateway_response' => $gatewayResponse,
            'paid_at' => now(),
            'invoice_number' => $this->generateInvoiceNumber(),
        ]);

        // Update related models
        if ($payment->payable instanceof Consultation) {
            // Consultation payment completed - no additional action needed
            // Commission will be calculated when consultation is completed
        } elseif ($payment->payable instanceof Subscription) {
            // Subscription payment completed
            $payment->payable->update(['status' => 'active']);
        }

        return $payment;
    }

    /**
     * Process payment failure
     */
    public function processPaymentFailure(Payment $payment, $reason = null, $gatewayResponse = [])
    {
        $payment->update([
            'status' => 'failed',
            'gateway_response' => $gatewayResponse,
        ]);

        // Handle failure scenarios
        if ($payment->payable instanceof Consultation) {
            // Cancel consultation if payment failed
            $payment->payable->update([
                'status' => 'cancelled',
                'cancelled_at' => now(),
                'cancellation_reason' => 'Payment failed: ' . $reason,
            ]);
        }

        return $payment;
    }

    /**
     * Generate invoice number
     */
    private function generateInvoiceNumber()
    {
        return 'INV-' . date('Ym') . '-' . str_pad(Payment::whereMonth('created_at', now()->month)->count() + 1, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate GST invoice data
     */
    public function generateInvoiceData(Payment $payment)
    {
        $payable = $payment->payable;
        
        if ($payable instanceof Consultation) {
            $professional = $payable->professional;
            $client = $payable->client;
            
            $lineItems = [
                [
                    'description' => 'Consultation Fee',
                    'amount' => $payable->consultation_fee,
                    'gst_rate' => 18,
                ],
                [
                    'description' => 'Platform Fee',
                    'amount' => $payable->platform_fee,
                    'gst_rate' => 18,
                ]
            ];
            
            $serviceProvider = [
                'name' => $professional->user->name,
                'profession' => $professional->profession_type,
                'license' => $professional->license_number,
            ];
            
            $customer = [
                'name' => $client->user->name,
                'profession' => $client->profession_type,
                'email' => $client->user->email,
                'phone' => $client->user->phone,
            ];
            
        } elseif ($payable instanceof Subscription) {
            $lineItems = [
                [
                    'description' => 'Subscription: ' . $payable->subscriptionPlan->name,
                    'amount' => $payable->amount_paid,
                    'gst_rate' => 18,
                ]
            ];
            
            $serviceProvider = [
                'name' => 'Medex Platform',
                'profession' => 'Technology Services',
                'license' => 'MEDEX001',
            ];
            
            $customer = [
                'name' => $payable->professional->user->name,
                'profession' => $payable->professional->profession_type,
                'email' => $payable->professional->user->email,
                'phone' => $payable->professional->user->phone,
            ];
        }

        return [
            'invoice_number' => $payment->invoice_number,
            'invoice_date' => $payment->paid_at->format('Y-m-d'),
            'payment_id' => $payment->payment_id,
            'line_items' => $lineItems,
            'subtotal' => $payment->amount,
            'gst_amount' => $payment->gst_amount,
            'total_amount' => $payment->total_amount,
            'service_provider' => $serviceProvider,
            'customer' => $customer,
            'payment_method' => 'Online Payment',
            'currency' => $payment->currency,
        ];
    }

    /**
     * Get payment analytics
     */
    public function getPaymentAnalytics($startDate = null, $endDate = null)
    {
        $query = Payment::where('status', 'completed');
        
        if ($startDate) {
            $query->where('paid_at', '>=', $startDate);
        }
        
        if ($endDate) {
            $query->where('paid_at', '<=', $endDate);
        }

        $totalRevenue = $query->sum('total_amount');
        $totalTransactions = $query->count();
        $averageTransaction = $totalTransactions > 0 ? $totalRevenue / $totalTransactions : 0;

        $consultationRevenue = $query->where('payment_type', 'consultation')->sum('total_amount');
        $subscriptionRevenue = $query->where('payment_type', 'subscription')->sum('total_amount');

        $monthlyRevenue = Payment::selectRaw('YEAR(paid_at) as year, MONTH(paid_at) as month, SUM(total_amount) as revenue')
            ->where('status', 'completed')
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->limit(12)
            ->get();

        return [
            'total_revenue' => $totalRevenue,
            'total_transactions' => $totalTransactions,
            'average_transaction' => round($averageTransaction, 2),
            'consultation_revenue' => $consultationRevenue,
            'subscription_revenue' => $subscriptionRevenue,
            'monthly_revenue' => $monthlyRevenue,
        ];
    }

    /**
     * Process refund
     */
    public function processRefund(Payment $payment, $refundAmount, $reason)
    {
        if ($payment->status !== 'completed') {
            throw new Exception('Cannot refund payment that is not completed');
        }

        if ($refundAmount > $payment->total_amount) {
            throw new Exception('Refund amount cannot exceed payment amount');
        }

        // In production, integrate with payment gateway for actual refund
        $this->processGatewayRefund($payment, $refundAmount);

        $payment->update([
            'status' => 'refunded',
            'refund_amount' => $refundAmount,
            'refund_reason' => $reason,
            'refunded_at' => now(),
        ]);

        return $payment;
    }

    /**
     * Mock gateway refund process
     */
    private function processGatewayRefund(Payment $payment, $refundAmount)
    {
        // This would integrate with actual payment gateway
        // For Razorpay:
        /*
        $api = new \Razorpay\Api\Api($keyId, $keySecret);
        $payment_gateway = $api->payment->fetch($payment->gateway_payment_id);
        $refund = $payment_gateway->refund(['amount' => $refundAmount * 100]);
        return $refund;
        */
        
        return true; // Mock success
    }
}