<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Professional extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'profession_type',
        'license_number',
        'membership_id',
        'registration_body',
        'license_expiry',
        'specializations',
        'experience_years',
        'consultation_fee',
        'availability',
        'office_address',
        'city',
        'state',
        'pincode',
        'documents',
        'verification_status',
        'verified_at',
        'verified_by',
        'rejection_reason',
        'is_featured',
        'rating',
        'total_consultations',
    ];

    protected $casts = [
        'specializations' => 'array',
        'availability' => 'array',
        'documents' => 'array',
        'license_expiry' => 'date',
        'verified_at' => 'datetime',
        'consultation_fee' => 'decimal:2',
        'rating' => 'decimal:2',
        'is_featured' => 'boolean',
    ];

    /**
     * Get the user that owns the professional profile
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the current active subscription
     */
    public function activeSubscription()
    {
        return $this->hasOne(Subscription::class)
                    ->where('status', 'active')
                    ->where('end_date', '>=', now());
    }

    /**
     * Get all subscriptions
     */
    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Get consultations for this professional
     */
    public function consultations()
    {
        return $this->hasMany(Consultation::class);
    }

    /**
     * Get commissions earned
     */
    public function commissions()
    {
        return $this->hasMany(Commission::class);
    }

    /**
     * Get payouts
     */
    public function payouts()
    {
        return $this->hasMany(Payout::class);
    }

    /**
     * Get reviews received
     */
    public function reviews()
    {
        return $this->hasMany(Review::class, 'to_user_id', 'user_id')
                    ->where('approved_by_admin', true);
    }

    /**
     * Scope for verified professionals
     */
    public function scopeVerified($query)
    {
        return $query->where('verification_status', 'verified');
    }

    /**
     * Scope for featured professionals
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Check if professional has active subscription
     */
    public function hasActiveSubscription()
    {
        return $this->activeSubscription()->exists();
    }
}
