<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Payout extends Model
{
    use HasFactory;

    protected $fillable = [
        'payout_id',
        'professional_id',
        'amount',
        'tds_amount',
        'net_amount',
        'status',
        'request_date',
        'processed_date',
        'bank_account_number',
        'bank_ifsc',
        'bank_name',
        'account_holder_name',
        'transaction_reference',
        'notes',
        'processed_by',
        'commission_ids',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'tds_amount' => 'decimal:2',
        'net_amount' => 'decimal:2',
        'request_date' => 'date',
        'processed_date' => 'date',
        'commission_ids' => 'array',
    ];

    /**
     * Get the professional that requested this payout
     */
    public function professional()
    {
        return $this->belongsTo(Professional::class);
    }

    /**
     * Get the admin who processed this payout
     */
    public function processedBy()
    {
        return $this->belongsTo(User::class, 'processed_by');
    }

    /**
     * Get commissions included in this payout
     */
    public function commissions()
    {
        if (empty($this->commission_ids)) {
            return collect();
        }
        return Commission::whereIn('id', $this->commission_ids)->get();
    }

    /**
     * Check if payout is completed
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Calculate TDS amount (typically 10% for professionals)
     */
    public function calculateTds($tdsRate = 10)
    {
        return round(($this->amount * $tdsRate) / 100, 2);
    }

    /**
     * Generate unique payout ID
     */
    public static function generatePayoutId()
    {
        return 'PAYOUT-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -6));
    }

    /**
     * Scope for pending payouts
     */
    public function scopePending($query)
    {
        return $query->where('status', 'requested');
    }

    /**
     * Scope for completed payouts
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}
