<?php

namespace App\Http\Controllers;

use App\Models\Review;
use App\Models\Consultation;
use App\Models\Professional;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReviewController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show form to create review for a consultation
     */
    public function create(Consultation $consultation)
    {
        $user = Auth::user();
        
        // Check if user can review this consultation
        if (!$this->canUserReviewConsultation($user, $consultation)) {
            abort(403, 'You cannot review this consultation.');
        }

        // Check if consultation is completed
        if ($consultation->status !== 'completed') {
            return redirect()->back()->with('error', 'You can only review completed consultations.');
        }

        // Check if review already exists
        $existingReview = Review::where('consultation_id', $consultation->id)
                              ->where('reviewer_id', $user->id)
                              ->first();

        if ($existingReview) {
            return redirect()->route('review.show', $existingReview)
                           ->with('info', 'You have already reviewed this consultation.');
        }

        return view('reviews.create', compact('consultation'));
    }

    /**
     * Store a review
     */
    public function store(Request $request, Consultation $consultation)
    {
        $user = Auth::user();
        
        if (!$this->canUserReviewConsultation($user, $consultation)) {
            abort(403, 'You cannot review this consultation.');
        }

        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'required|string|min:10|max:1000',
            'is_anonymous' => 'boolean'
        ]);

        // Check if review already exists
        $existingReview = Review::where('consultation_id', $consultation->id)
                              ->where('reviewer_id', $user->id)
                              ->first();

        if ($existingReview) {
            return redirect()->route('review.show', $existingReview)
                           ->with('error', 'You have already reviewed this consultation.');
        }

        // Determine reviewee (the person being reviewed)
        if ($user->user_type === 'client') {
            $revieweeId = $consultation->professional->user_id;
        } else {
            $revieweeId = $consultation->client->user_id;
        }

        DB::transaction(function () use ($request, $consultation, $user, $revieweeId) {
            // Create the review
            $review = Review::create([
                'consultation_id' => $consultation->id,
                'reviewer_id' => $user->id,
                'reviewee_id' => $revieweeId,
                'rating' => $request->rating,
                'comment' => $request->comment,
                'is_anonymous' => $request->boolean('is_anonymous'),
                'status' => 'active' // Auto-approve for now; can add moderation later
            ]);

            // Update professional's average rating if they're being reviewed
            if ($user->user_type === 'client') {
                $this->updateProfessionalRating($consultation->professional);
            }
        });

        return redirect()->route('consultation.show', $consultation)
                        ->with('success', 'Thank you for your review!');
    }

    /**
     * Show a specific review
     */
    public function show(Review $review)
    {
        $user = Auth::user();
        
        // Check if user can view this review
        if ($user->id !== $review->reviewer_id && 
            $user->id !== $review->reviewee_id && 
            $user->user_type !== 'admin') {
            abort(403, 'You cannot view this review.');
        }

        $review->load(['consultation', 'reviewer', 'reviewee']);

        return view('reviews.show', compact('review'));
    }

    /**
     * Update a review (only reviewer can update within 24 hours)
     */
    public function update(Request $request, Review $review)
    {
        $user = Auth::user();
        
        if ($user->id !== $review->reviewer_id) {
            abort(403, 'You can only edit your own reviews.');
        }

        // Check if review can be edited (within 24 hours)
        if ($review->created_at->diffInHours(now()) > 24) {
            return redirect()->back()->with('error', 'Reviews can only be edited within 24 hours.');
        }

        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'required|string|min:10|max:1000',
            'is_anonymous' => 'boolean'
        ]);

        DB::transaction(function () use ($request, $review) {
            $review->update([
                'rating' => $request->rating,
                'comment' => $request->comment,
                'is_anonymous' => $request->boolean('is_anonymous')
            ]);

            // Update professional's rating if applicable
            if ($review->reviewer->user_type === 'client') {
                $this->updateProfessionalRating($review->consultation->professional);
            }
        });

        return redirect()->route('review.show', $review)
                        ->with('success', 'Review updated successfully!');
    }

    /**
     * Show reviews for a professional
     */
    public function professionalReviews(Professional $professional, Request $request)
    {
        $query = Review::where('reviewee_id', $professional->user_id)
                      ->where('status', 'active')
                      ->with(['reviewer', 'consultation']);

        // Filter by rating if specified
        if ($request->filled('rating')) {
            $query->where('rating', $request->rating);
        }

        // Sort by latest first
        $query->orderBy('created_at', 'desc');

        $reviews = $query->paginate(10);

        // Calculate rating statistics
        $ratingStats = Review::where('reviewee_id', $professional->user_id)
                           ->where('status', 'active')
                           ->selectRaw('
                               AVG(rating) as average_rating,
                               COUNT(*) as total_reviews,
                               SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as five_star,
                               SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as four_star,
                               SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as three_star,
                               SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as two_star,
                               SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as one_star
                           ')
                           ->first();

        return view('reviews.professional', compact('professional', 'reviews', 'ratingStats'));
    }

    /**
     * Show user's given reviews
     */
    public function myReviews(Request $request)
    {
        $user = Auth::user();
        
        $query = Review::where('reviewer_id', $user->id)
                      ->with(['consultation', 'reviewee']);

        if ($request->filled('rating')) {
            $query->where('rating', $request->rating);
        }

        $reviews = $query->orderBy('created_at', 'desc')->paginate(10);

        return view('reviews.my-reviews', compact('reviews'));
    }

    /**
     * Show reviews received by user
     */
    public function receivedReviews(Request $request)
    {
        $user = Auth::user();
        
        $query = Review::where('reviewee_id', $user->id)
                      ->where('status', 'active')
                      ->with(['consultation', 'reviewer']);

        if ($request->filled('rating')) {
            $query->where('rating', $request->rating);
        }

        $reviews = $query->orderBy('created_at', 'desc')->paginate(10);

        return view('reviews.received', compact('reviews'));
    }

    /**
     * Admin: Show all reviews for moderation
     */
    public function adminReviews(Request $request)
    {
        $user = Auth::user();
        
        if ($user->user_type !== 'admin') {
            abort(403, 'Admin access required.');
        }

        $query = Review::with(['reviewer', 'reviewee', 'consultation']);

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('rating')) {
            $query->where('rating', $request->rating);
        }

        $reviews = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.reviews', compact('reviews'));
    }

    /**
     * Admin: Update review status
     */
    public function updateStatus(Request $request, Review $review)
    {
        $user = Auth::user();
        
        if ($user->user_type !== 'admin') {
            abort(403, 'Admin access required.');
        }

        $request->validate([
            'status' => 'required|in:active,hidden,flagged'
        ]);

        $review->update(['status' => $request->status]);

        return redirect()->back()->with('success', 'Review status updated successfully.');
    }

    /**
     * Report a review
     */
    public function report(Request $request, Review $review)
    {
        $user = Auth::user();
        
        $request->validate([
            'reason' => 'required|string|max:500'
        ]);

        // Check if user already reported this review
        $existingReport = DB::table('review_reports')
                           ->where('review_id', $review->id)
                           ->where('reporter_id', $user->id)
                           ->exists();

        if ($existingReport) {
            return redirect()->back()->with('error', 'You have already reported this review.');
        }

        // Create report
        DB::table('review_reports')->insert([
            'review_id' => $review->id,
            'reporter_id' => $user->id,
            'reason' => $request->reason,
            'created_at' => now(),
            'updated_at' => now()
        ]);

        return redirect()->back()->with('success', 'Review has been reported for moderation.');
    }

    /**
     * Check if user can review a consultation
     */
    private function canUserReviewConsultation($user, $consultation)
    {
        if ($user->user_type === 'client') {
            return $user->id === $consultation->client->user_id;
        } elseif ($user->user_type === 'professional') {
            return $user->id === $consultation->professional->user_id;
        }
        
        return false;
    }

    /**
     * Update professional's average rating
     */
    private function updateProfessionalRating($professional)
    {
        $stats = Review::where('reviewee_id', $professional->user_id)
                      ->where('status', 'active')
                      ->selectRaw('AVG(rating) as avg_rating, COUNT(*) as total_reviews')
                      ->first();

        $professional->update([
            'average_rating' => round($stats->avg_rating, 2),
            'total_reviews' => $stats->total_reviews
        ]);
    }
}
