<?php

namespace App\Http\Controllers;

use App\Models\Client;
use App\Models\Professional;
use App\Models\Consultation;
use App\Services\ConsultationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ClientController extends Controller
{
    protected $consultationService;

    public function __construct(ConsultationService $consultationService)
    {
        $this->middleware('auth');
        $this->consultationService = $consultationService;
    }

    public function dashboard()
    {
        $user = Auth::user();
        $client = $user->client;
        
        // Statistics
        $totalConsultations = Consultation::where('client_id', $client->id)->count();
        $upcomingConsultations = Consultation::where('client_id', $client->id)
            ->where('scheduled_at', '>', now())
            ->where('status', 'confirmed')
            ->count();
        
        $completedConsultations = Consultation::where('client_id', $client->id)
            ->where('status', 'completed')
            ->count();
            
        $totalSpent = Consultation::where('client_id', $client->id)
            ->where('payment_status', 'completed')
            ->sum('total_amount');
        
        // Recent consultations
        $recentConsultations = Consultation::where('client_id', $client->id)
            ->with(['professional.user', 'service'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        // Health tips
        $healthTips = [
            "Regular health checkups can prevent major medical expenses.",
            "Maintain a healthy lifestyle to avoid chronic diseases.",
            "Keep your medical records organized for legal consultations.",
            "Consider health insurance to protect against financial burden.",
            "Early consultation can save both time and money."
        ];
        
        $randomTip = $healthTips[array_rand($healthTips)];
        
        return view('client.dashboard', compact(
            'totalConsultations',
            'upcomingConsultations', 
            'completedConsultations',
            'totalSpent',
            'recentConsultations',
            'randomTip'
        ));
    }

    /**
     * Calculate growth rate percentage
     */
    private function calculateGrowthRate($previous, $current)
    {
        if ($previous == 0) return $current > 0 ? 100 : 0;
        return round((($current - $previous) / $previous) * 100, 1);
    }

    /**
     * Get favorite professionals count
     */
    private function getFavoriteProfessionalsCount($client)
    {
        return $client->consultations()
            ->select('professional_id')
            ->groupBy('professional_id')
            ->having(DB::raw('count(*)'), '>', 1)
            ->count();
    }

    /**
     * Get favorite professionals
     */
    private function getFavoriteProfessionals($client)
    {
        return Professional::with(['user'])
            ->whereIn('id', function($query) use ($client) {
                $query->select('professional_id')
                      ->from('consultations')
                      ->where('client_id', $client->id)
                      ->groupBy('professional_id')
                      ->having(DB::raw('count(*)'), '>', 1);
            })
            ->limit(5)
            ->get();
    }

    /**
     * Show client profile
     */
    public function profile()
    {
        $user = Auth::user();
        $client = $user->client;

        return view('client.profile', compact('client'));
    }

    /**
     * Update client profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        $client = $user->client;

        $request->validate([
            'clinic_hospital_name' => 'nullable|string|max:255',
            'medical_license' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:500',
            'bio' => 'nullable|string|max:1000',
        ]);

        $client->update([
            'clinic_hospital_name' => $request->clinic_hospital_name,
            'medical_license' => $request->medical_license,
            'address' => $request->address,
            'bio' => $request->bio,
        ]);

        return back()->with('success', 'Profile updated successfully!');
    }

    /**
     * Search professionals
     */
    public function searchProfessionals(Request $request)
    {
        $query = Professional::with(['user', 'activeSubscription.subscriptionPlan'])
            ->where('verification_status', 'verified');

        // Filter by profession type
        if ($request->filled('profession_type')) {
            $query->where('profession_type', $request->profession_type);
        }

        // Filter by city
        if ($request->filled('city')) {
            $query->where('city', 'like', '%' . $request->city . '%');
        }

        // Filter by consultation fee range
        if ($request->filled('min_fee')) {
            $query->where('consultation_fee', '>=', $request->min_fee);
        }
        if ($request->filled('max_fee')) {
            $query->where('consultation_fee', '<=', $request->max_fee);
        }

        // Filter by specializations
        if ($request->filled('specialization')) {
            $query->whereJsonContains('specializations', $request->specialization);
        }

        // Sort options
        $sortBy = $request->get('sort_by', 'rating');
        switch ($sortBy) {
            case 'fee_low':
                $query->orderBy('consultation_fee', 'asc');
                break;
            case 'fee_high':
                $query->orderBy('consultation_fee', 'desc');
                break;
            case 'experience':
                $query->orderBy('experience_years', 'desc');
                break;
            case 'rating':
            default:
                $query->orderBy('rating', 'desc');
                break;
        }

        // Prioritize featured professionals
        $query->orderBy('is_featured', 'desc');

        $professionals = $query->paginate(12);

        // Get filter options
        $professionTypes = [
            'advocate' => 'Advocate (Lawyer)',
            'chartered_accountant' => 'Chartered Accountant (CA)',
            'company_secretary' => 'Company Secretary (CS)',
            'cost_management_accountant' => 'Cost & Management Accountant (CMA)',
        ];

        return view('client.professionals', compact('professionals', 'professionTypes'));
    }

    /**
     * Show consultation booking form
     */
    public function bookConsultation(Professional $professional)
    {
        if ($professional->verification_status !== 'verified') {
            return back()->with('error', 'This professional is not verified yet.');
        }

        if (!$professional->hasActiveSubscription()) {
            return back()->with('error', 'This professional is not available for consultations.');
        }

        return view('client.book-consultation', compact('professional'));
    }

    /**
     * Store consultation booking
     */
    public function storeBooking(Request $request, Professional $professional)
    {
        $user = Auth::user();
        $client = $user->client;

        $request->validate([
            'consultation_type' => 'required|in:video,audio,chat',
            'date' => 'required|date|after:today',
            'time_slot' => 'required',
            'notes' => 'nullable|string|max:500',
        ]);

        // Combine date and time
        $scheduledAt = $request->date . ' ' . $request->time_slot;

        // Check if slot is available
        $availableSlots = $this->consultationService->getAvailableSlots($professional, $request->date);
        $isSlotAvailable = collect($availableSlots)->contains('time', $request->time_slot);

        if (!$isSlotAvailable) {
            return back()->with('error', 'Selected time slot is no longer available.');
        }

        try {
            $consultation = $this->consultationService->createConsultation($client, $professional, [
                'consultation_type' => $request->consultation_type,
                'scheduled_at' => $scheduledAt,
                'notes' => $request->notes,
            ]);

            // Redirect to payment page
            return redirect()->route('consultation.payment', $consultation)
                           ->with('success', 'Consultation booked successfully! Please complete the payment.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to book consultation. Please try again.');
        }
    }

    /**
     * Show client consultations
     */
    public function consultations(Request $request)
    {
        $user = Auth::user();
        $client = $user->client;

        $status = $request->get('status', 'all');
        
        $consultationsQuery = $client->consultations()->with(['professional.user']);

        if ($status !== 'all') {
            $consultationsQuery->where('status', $status);
        }

        $consultations = $consultationsQuery->orderBy('scheduled_at', 'desc')->paginate(10);

        return view('client.consultations', compact('consultations', 'status'));
    }

    /**
     * Get available time slots for AJAX
     */
    public function getAvailableSlots(Request $request, Professional $professional)
    {
        $request->validate([
            'date' => 'required|date|after:today',
        ]);

        $slots = $this->consultationService->getAvailableSlots($professional, $request->date);

        return response()->json(['slots' => $slots]);
    }
}
