<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Professional;
use App\Models\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Show registration form
     */
    public function showRegisterForm()
    {
        return view('auth.register');
    }

    /**
     * Handle registration
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'required|string|max:15',
            'user_type' => 'required|in:professional,client',
            'profession_type' => 'required_if:user_type,professional|in:advocate,chartered_accountant,company_secretary,cost_management_accountant',
            'client_profession' => 'required_if:user_type,client|in:doctor,nurse,pharmacist,clinic_owner,hospital_admin,medical_store_owner',
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:255',
            'pincode' => 'required|string|size:6',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Create user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
            'user_type' => $request->user_type,
            'status' => $request->user_type === 'professional' ? 'pending_approval' : 'active',
        ]);

        // Create professional or client profile
        if ($request->user_type === 'professional') {
            Professional::create([
                'user_id' => $user->id,
                'profession_type' => $request->profession_type,
                'city' => $request->city,
                'state' => $request->state,
                'pincode' => $request->pincode,
                'verification_status' => 'pending',
            ]);
        } else {
            Client::create([
                'user_id' => $user->id,
                'profession_type' => $request->client_profession,
                'city' => $request->city,
                'state' => $request->state,
                'pincode' => $request->pincode,
                'verification_status' => 'verified', // Auto-verify clients
                'verified_at' => now(),
            ]);
        }

        Auth::login($user);

        // Redirect based on user type
        if ($user->user_type === 'professional') {
            return redirect()->route('professional.dashboard')->with('success', 'Registration successful! Your account is pending approval.');
        } else {
            return redirect()->route('client.dashboard')->with('success', 'Registration successful! Welcome to Medex.');
        }
    }

    /**
     * Show login form
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle login
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (!Auth::attempt($credentials, $request->boolean('remember'))) {
            throw ValidationException::withMessages([
                'email' => trans('auth.failed'),
            ]);
        }

        $request->session()->regenerate();

        $user = Auth::user();

        // Redirect based on user type
        if ($user->user_type === 'admin') {
            return redirect()->intended(route('admin.dashboard'));
        } elseif ($user->user_type === 'professional') {
            return redirect()->intended(route('professional.dashboard'));
        } else {
            return redirect()->intended(route('client.dashboard'));
        }
    }

    /**
     * Handle logout
     */
    public function logout(Request $request)
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home');
    }
}
