<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Professional;
use App\Models\Consultation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ProfessionalController extends Controller
{
    /**
     * Get professional dashboard data
     */
    public function dashboard(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            // Statistics
            $stats = [
                'total_consultations' => $professional->consultations()->count(),
                'consultations_this_month' => $professional->consultations()
                    ->whereMonth('scheduled_at', now()->month)
                    ->count(),
                'total_earnings' => $professional->consultations()
                    ->where('payment_status', 'completed')
                    ->sum('professional_fee'),
                'earnings_this_month' => $professional->consultations()
                    ->where('payment_status', 'completed')
                    ->whereMonth('scheduled_at', now()->month)
                    ->sum('professional_fee'),
                'average_rating' => $professional->reviews()->avg('rating') ?? 0,
                'pending_consultations' => $professional->consultations()
                    ->where('status', 'scheduled')
                    ->where('scheduled_at', '>=', now())
                    ->count(),
            ];

            // Recent consultations
            $recentConsultations = $professional->consultations()
                ->with(['client.user'])
                ->latest()
                ->limit(5)
                ->get();

            return response()->json([
                'status' => 'success',
                'data' => [
                    'stats' => $stats,
                    'recent_consultations' => $recentConsultations
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get dashboard data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get professional profile
     */
    public function profile(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'data' => [
                    'user' => $user,
                    'professional' => $professional
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update professional profile
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'consultation_fee' => 'sometimes|numeric|min:0',
                'experience_years' => 'sometimes|integer|min:0|max:50',
                'specializations' => 'sometimes|array',
                'office_address' => 'sometimes|string|max:500',
                'bio' => 'sometimes|string|max:1000',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $professional->update($request->only([
                'consultation_fee',
                'experience_years',
                'specializations',
                'office_address',
                'bio'
            ]));

            return response()->json([
                'status' => 'success',
                'message' => 'Profile updated successfully',
                'data' => [
                    'professional' => $professional->fresh()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update profile',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get consultations
     */
    public function consultations(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            $query = $professional->consultations()->with(['client.user']);

            // Filter by status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Filter by date range
            if ($request->filled('from_date')) {
                $query->whereDate('scheduled_at', '>=', $request->from_date);
            }

            if ($request->filled('to_date')) {
                $query->whereDate('scheduled_at', '<=', $request->to_date);
            }

            $consultations = $query->orderBy('scheduled_at', 'desc')
                ->paginate($request->get('per_page', 15));

            return response()->json([
                'status' => 'success',
                'data' => $consultations
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get consultations',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get earnings
     */
    public function earnings(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            $query = $professional->consultations()
                ->where('payment_status', 'completed');

            // Filter by date range
            if ($request->filled('from_date')) {
                $query->whereDate('scheduled_at', '>=', $request->from_date);
            }

            if ($request->filled('to_date')) {
                $query->whereDate('scheduled_at', '<=', $request->to_date);
            }

            $earnings = [
                'total_earnings' => $query->sum('professional_fee'),
                'total_consultations' => $query->count(),
                'average_fee' => $query->avg('professional_fee'),
                'monthly_breakdown' => $professional->consultations()
                    ->where('payment_status', 'completed')
                    ->selectRaw('YEAR(scheduled_at) as year, MONTH(scheduled_at) as month, SUM(professional_fee) as earnings, COUNT(*) as consultations')
                    ->groupByRaw('YEAR(scheduled_at), MONTH(scheduled_at)')
                    ->orderByRaw('YEAR(scheduled_at) DESC, MONTH(scheduled_at) DESC')
                    ->limit(12)
                    ->get()
            ];

            return response()->json([
                'status' => 'success',
                'data' => $earnings
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get earnings',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get subscription details
     */
    public function subscription(Request $request)
    {
        try {
            $user = $request->user();
            $professional = $user->professional;

            if (!$professional) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Professional profile not found'
                ], 404);
            }

            $subscription = $professional->activeSubscription;

            return response()->json([
                'status' => 'success',
                'data' => [
                    'subscription' => $subscription ? $subscription->load('subscriptionPlan') : null,
                    'has_active_subscription' => $subscription ? true : false
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get subscription details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upgrade subscription
     */
    public function upgradeSubscription(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'plan_id' => 'required|exists:subscription_plans,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // This would typically integrate with payment processing
            // For now, we'll return a success message indicating the upgrade is pending
            
            return response()->json([
                'status' => 'success',
                'message' => 'Subscription upgrade initiated. Please complete payment.',
                'data' => [
                    'plan_id' => $request->plan_id,
                    'next_step' => 'payment_required'
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to upgrade subscription',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}