<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Consultation;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PaymentController extends Controller
{
    /**
     * Process consultation payment
     */
    public function processConsultationPayment(Request $request, Consultation $consultation)
    {
        try {
            $user = $request->user();

            // Check if user is the client for this consultation
            if ($user->role !== 'client' || $consultation->client_id !== $user->client->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // Check if payment already completed
            if ($consultation->payment_status === 'completed') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Payment already completed'
                ], 400);
            }

            $validator = Validator::make($request->all(), [
                'payment_method' => 'required|in:card,upi,netbanking,wallet',
                'payment_gateway' => 'required|in:razorpay,stripe,payu',
                'gateway_transaction_id' => 'required|string',
                'gateway_payment_id' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Create payment record
            $payment = Payment::create([
                'user_id' => $user->id,
                'consultation_id' => $consultation->id,
                'amount' => $consultation->total_amount,
                'currency' => 'INR',
                'payment_method' => $request->payment_method,
                'payment_gateway' => $request->payment_gateway,
                'gateway_transaction_id' => $request->gateway_transaction_id,
                'gateway_payment_id' => $request->gateway_payment_id,
                'status' => 'completed', // In real scenario, this would be verified with gateway
                'transaction_date' => now(),
                'gst_amount' => $consultation->gst_amount,
                'platform_fee' => $consultation->platform_fee,
                'professional_fee' => $consultation->professional_fee,
            ]);

            // Update consultation payment status
            $consultation->update([
                'payment_status' => 'completed',
                'status' => 'confirmed'
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Payment processed successfully',
                'data' => [
                    'payment' => $payment,
                    'consultation' => $consultation->fresh()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Payment processing failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process subscription payment
     */
    public function processSubscriptionPayment(Request $request, Subscription $subscription)
    {
        try {
            $user = $request->user();

            // Check if user owns this subscription
            if ($subscription->user_id !== $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            $validator = Validator::make($request->all(), [
                'payment_method' => 'required|in:card,upi,netbanking,wallet',
                'payment_gateway' => 'required|in:razorpay,stripe,payu',
                'gateway_transaction_id' => 'required|string',
                'gateway_payment_id' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $subscriptionPlan = $subscription->subscriptionPlan;
            $amount = $subscriptionPlan->price;
            $gstAmount = $amount * 0.18; // 18% GST
            $totalAmount = $amount + $gstAmount;

            // Create payment record
            $payment = Payment::create([
                'user_id' => $user->id,
                'subscription_id' => $subscription->id,
                'amount' => $totalAmount,
                'currency' => 'INR',
                'payment_method' => $request->payment_method,
                'payment_gateway' => $request->payment_gateway,
                'gateway_transaction_id' => $request->gateway_transaction_id,
                'gateway_payment_id' => $request->gateway_payment_id,
                'status' => 'completed',
                'transaction_date' => now(),
                'gst_amount' => $gstAmount,
            ]);

            // Update subscription status
            $subscription->update([
                'status' => 'active',
                'starts_at' => now(),
                'expires_at' => now()->addDays($subscriptionPlan->duration_days),
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Subscription payment processed successfully',
                'data' => [
                    'payment' => $payment,
                    'subscription' => $subscription->fresh()->load('subscriptionPlan')
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Subscription payment processing failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment history
     */
    public function paymentHistory(Request $request)
    {
        try {
            $user = $request->user();

            $query = $user->payments()->with(['consultation', 'subscription.subscriptionPlan']);

            // Filter by type
            if ($request->filled('type')) {
                if ($request->type === 'consultation') {
                    $query->whereNotNull('consultation_id');
                } elseif ($request->type === 'subscription') {
                    $query->whereNotNull('subscription_id');
                }
            }

            // Filter by status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Filter by date range
            if ($request->filled('from_date')) {
                $query->whereDate('transaction_date', '>=', $request->from_date);
            }

            if ($request->filled('to_date')) {
                $query->whereDate('transaction_date', '<=', $request->to_date);
            }

            $payments = $query->orderBy('transaction_date', 'desc')
                ->paginate($request->get('per_page', 15));

            return response()->json([
                'status' => 'success',
                'data' => $payments
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to get payment history',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download invoice
     */
    public function downloadInvoice(Request $request, Payment $payment)
    {
        try {
            $user = $request->user();

            // Check if user owns this payment
            if ($payment->user_id !== $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // In a real application, this would generate and return a PDF invoice
            // For now, we'll return the payment details in a structured format
            
            $invoiceData = [
                'invoice_number' => 'INV-' . str_pad($payment->id, 6, '0', STR_PAD_LEFT),
                'invoice_date' => $payment->transaction_date->format('Y-m-d'),
                'payment' => $payment->load(['consultation.professional.user', 'subscription.subscriptionPlan']),
                'billing_address' => [
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                ],
                'company_details' => [
                    'name' => 'Medex Legal & Financial Consultation Platform',
                    'address' => 'Your Company Address',
                    'gst_number' => 'YOUR_GST_NUMBER',
                ],
            ];

            return response()->json([
                'status' => 'success',
                'data' => [
                    'invoice' => $invoiceData,
                    'download_url' => route('payment.invoice', $payment->id) // Web route for PDF download
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to generate invoice',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Request refund
     */
    public function requestRefund(Request $request, Payment $payment)
    {
        try {
            $user = $request->user();

            // Check if user owns this payment
            if ($payment->user_id !== $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized access'
                ], 403);
            }

            // Check if refund is already requested or processed
            if (in_array($payment->status, ['refund_requested', 'refunded'])) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Refund already requested or processed'
                ], 400);
            }

            $validator = Validator::make($request->all(), [
                'reason' => 'required|string|max:500',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Update payment status
            $payment->update([
                'status' => 'refund_requested',
                'refund_reason' => $request->reason,
                'refund_requested_at' => now(),
            ]);

            // If it's a consultation payment, update consultation status
            if ($payment->consultation_id) {
                $payment->consultation->update([
                    'status' => 'cancelled',
                    'payment_status' => 'refund_requested'
                ]);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Refund request submitted successfully',
                'data' => [
                    'payment' => $payment->fresh(),
                    'estimated_processing_time' => '3-7 business days'
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to request refund',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}